<?php
// helpers.php
// Full helper file for your USSD app (includes DB helpers + draft_update + JSON HTTP helpers)

declare(strict_types=1);

/**
 * -------------------------------
 * USSD BASIC HELPERS
 * -------------------------------
 */
function ussd_clean($s): string
{
    $s = (string)$s;
    $s = trim($s);
    $s = str_replace(["\r", "\n", "\t"], " ", $s);
    return $s;
}

/**
 * Parses gateway text like: "1*John*1990-01-01*1"
 * Returns array of inputs in order.
 */
function ussd_parse_text(string $text): array
{
    $text = trim($text);
    if ($text === "") return [];

    $parts = explode("*", $text);
    $out = [];
    foreach ($parts as $p) {
        $p = trim((string)$p);
        // Keep empty parts out (prevents odd issues)
        if ($p !== "") $out[] = $p;
    }
    return $out;
}

/**
 * Sends USSD response and exits
 */
function ussd_response(string $type, string $message): void
{
    header("Content-Type: text/plain; charset=utf-8");
    echo $type . " " . $message;
    exit;
}

/**
 * -------------------------------
 * HTTP JSON HELPERS (cURL)
 * - Returns decoded array
 * - If JSON invalid: returns ok=false + raw_snippet + http_code
 * -------------------------------
 */
function get_json(string $url, int $timeout = 20): array
{
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT        => $timeout,
        CURLOPT_HTTPHEADER     => [
            "Accept: application/json"
        ],
    ]);

    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($resp === false) {
        return ["ok" => false, "error" => "cURL error: " . $err, "http_code" => $code];
    }

    $decoded = json_decode($resp, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return [
            "ok" => false,
            "error" => "Invalid JSON from API",
            "http_code" => $code,
            "raw_snippet" => mb_substr((string)$resp, 0, 600)
        ];
    }

    if (is_array($decoded) && !isset($decoded["http_code"])) $decoded["http_code"] = $code;
    return is_array($decoded) ? $decoded : ["ok" => false, "error" => "API returned non-array JSON", "http_code" => $code];
}

function post_json(string $url, array $data, int $timeout = 20): array
{
    $payload = json_encode($data);

    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_POST           => true,
        CURLOPT_HTTPHEADER     => [
            "Content-Type: application/json",
            "Accept: application/json"
        ],
        CURLOPT_POSTFIELDS     => $payload,
        CURLOPT_CONNECTTIMEOUT => 10,
        CURLOPT_TIMEOUT        => $timeout,
    ]);

    $resp = curl_exec($ch);
    $err  = curl_error($ch);
    $code = (int)curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if ($resp === false) {
        return ["ok" => false, "error" => "cURL error: " . $err, "http_code" => $code];
    }

    $decoded = json_decode($resp, true);
    if (json_last_error() !== JSON_ERROR_NONE) {
        return [
            "ok" => false,
            "error" => "Invalid JSON from API",
            "http_code" => $code,
            "raw_snippet" => mb_substr((string)$resp, 0, 600)
        ];
    }

    if (is_array($decoded) && !isset($decoded["http_code"])) $decoded["http_code"] = $code;
    return is_array($decoded) ? $decoded : ["ok" => false, "error" => "API returned non-array JSON", "http_code" => $code];
}

/**
 * -------------------------------
 * DATABASE HELPERS (MySQLi)
 * Assumes database.php defines: $con (mysqli)
 * -------------------------------
 */

/**
 * Internal: bind params safely
 */
function _db_bind_params(mysqli_stmt $stmt, string $types, array $params): void
{
    if ($types === "" || empty($params)) return;

    // mysqli requires references
    $refs = [];
    foreach ($params as $k => $v) {
        $refs[$k] = $params[$k];
    }
    $bindParams = array_merge([$types], $refs);

    // Convert to references array
    $tmp = [];
    foreach ($bindParams as $key => $value) {
        $tmp[$key] = &$bindParams[$key];
    }

    call_user_func_array([$stmt, 'bind_param'], $tmp);
}

/**
 * Execute INSERT/UPDATE/DELETE
 * Returns: [ok=>bool, affected=>int, insert_id=>int, error=>string]
 */
function db_exec(mysqli $con, string $sql, string $types = "", array $params = []): array
{
    $stmt = $con->prepare($sql);
    if (!$stmt) {
        return ["ok" => false, "error" => "Prepare failed: " . $con->error];
    }

    if ($types !== "" && !empty($params)) {
        _db_bind_params($stmt, $types, $params);
    }

    $ok = $stmt->execute();
    if (!$ok) {
        $err = $stmt->error ?: "Execute failed";
        $stmt->close();
        return ["ok" => false, "error" => $err];
    }

    $affected = $stmt->affected_rows;
    $insertId = $stmt->insert_id;
    $stmt->close();

    return ["ok" => true, "affected" => $affected, "insert_id" => $insertId];
}

/**
 * Fetch first row
 */
function db_fetch_one(mysqli $con, string $sql, string $types = "", array $params = []): ?array
{
    $stmt = $con->prepare($sql);
    if (!$stmt) return null;

    if ($types !== "" && !empty($params)) {
        _db_bind_params($stmt, $types, $params);
    }

    if (!$stmt->execute()) {
        $stmt->close();
        return null;
    }

    $res = $stmt->get_result();
    $row = $res ? $res->fetch_assoc() : null;

    $stmt->close();
    return $row ?: null;
}

/**
 * Fetch all rows
 */
function db_fetch_all(mysqli $con, string $sql, string $types = "", array $params = []): array
{
    $stmt = $con->prepare($sql);
    if (!$stmt) return [];

    if ($types !== "" && !empty($params)) {
        _db_bind_params($stmt, $types, $params);
    }

    if (!$stmt->execute()) {
        $stmt->close();
        return [];
    }

    $res = $stmt->get_result();
    $rows = [];
    if ($res) {
        while ($r = $res->fetch_assoc()) $rows[] = $r;
    }

    $stmt->close();
    return $rows;
}

/**
 * -------------------------------
 * DRAFT HELPERS (ussd_reg_drafts)
 * Updates only the provided columns for the session_id.
 * Usage: draft_update($con, $sessionId, ["step"=>"ASK_DOB","gender"=>"Male"]);
 * -------------------------------
 */
function draft_update(mysqli $con, string $sessionId, array $data): array
{
    if ($sessionId === "" || empty($data)) {
        return ["ok" => false, "error" => "draft_update: sessionId or data missing"];
    }

    // Build dynamic update statement
    $setParts = [];
    $types = "";
    $params = [];

    foreach ($data as $col => $val) {
        // Very basic column whitelist (optional). If you want strict security, list allowed columns here.
        if (!preg_match('/^[a-zA-Z0-9_]+$/', (string)$col)) {
            return ["ok" => false, "error" => "draft_update: invalid column {$col}"];
        }

        $setParts[] = "{$col}=?";
        $types .= "s";
        $params[] = (string)$val;
    }

    $types .= "s";
    $params[] = $sessionId;

    $sql = "UPDATE ussd_reg_drafts SET " . implode(", ", $setParts) . " WHERE session_id=?";

    return db_exec($con, $sql, $types, $params);
}