<?php
/**
 * claims_signup.php
 *
 * ✅ Updated to allow customer to initiate CLAIMS ANY TIME (duplicates allowed)
 * ✅ Always creates a NEW claim row (no duplicate check that returns existing claim)
 * ✅ claim_id will ALWAYS be unique (AUTO_INCREMENT in DB)
 * ✅ claim_number is generated uniquely per day using a MySQL named lock to avoid clashes
 *
 * Format: yymmdd + 5-digit sequence
 * Example: 26020900001
 */

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

header("Content-Type: application/json; charset=utf-8");

include("database.php"); // must create $con (mysqli)
date_default_timezone_set("Africa/Accra");

$path = $_SERVER['REQUEST_URI'] ?? 'claims_signup.php';

/* =========================
   LOG FUNCTION
   Table: users_logs(userindex, caption, logs_date, path, data_inf)
   - userindex will be mobile_money_number for this API
========================= */
function log_activity($con, $userindex, $caption, $path, $data_inf) {
    if (!isset($con) || !$con) return;

    $sql = "INSERT INTO users_logs (userindex, caption, logs_date, path, data_inf)
            VALUES (?, ?, NOW(), ?, ?)";

    if ($stmt = mysqli_prepare($con, $sql)) {
        $u = (string)$userindex;
        $c = (string)$caption;
        $p = (string)$path;
        $d = (string)$data_inf;

        mysqli_stmt_bind_param($stmt, "ssss", $u, $c, $p, $d);
        mysqli_stmt_execute($stmt);
        mysqli_stmt_close($stmt);
    }
}

function out($con, $userindex, $path, $ok, $extra = [], $caption = null) {
    if ($caption !== null) {
        log_activity($con, $userindex, $caption, $path, json_encode($extra, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
    }
    echo json_encode(array_merge(["ok" => (bool)$ok], $extra), JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

/**
 * Acquire a named lock (prevents two requests generating same claim_number at same time).
 * Returns true/false.
 */
function get_named_lock($con, $lockName, $timeoutSeconds = 5) {
    $sql = "SELECT GET_LOCK(?, ?) AS got_lock";
    $stmt = mysqli_prepare($con, $sql);
    if (!$stmt) return false;
    mysqli_stmt_bind_param($stmt, "si", $lockName, $timeoutSeconds);
    mysqli_stmt_execute($stmt);
    $res = mysqli_stmt_get_result($stmt);
    $row = $res ? mysqli_fetch_assoc($res) : null;
    mysqli_stmt_close($stmt);
    return (isset($row["got_lock"]) && (int)$row["got_lock"] === 1);
}

function release_named_lock($con, $lockName) {
    $sql = "SELECT RELEASE_LOCK(?) AS released";
    $stmt = mysqli_prepare($con, $sql);
    if (!$stmt) return false;
    mysqli_stmt_bind_param($stmt, "s", $lockName);
    mysqli_stmt_execute($stmt);
    mysqli_stmt_close($stmt);
    return true;
}

// ---------- METHOD CHECK ----------
if ($_SERVER["REQUEST_METHOD"] !== "POST") {
    log_activity($con, "UNKNOWN", "INVALID REQUEST METHOD", $path, json_encode([
        "method" => $_SERVER["REQUEST_METHOD"] ?? ""
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
    echo json_encode(["ok" => false, "error" => "Invalid request method"], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

if (!isset($con) || !$con) {
    echo json_encode(["ok" => false, "error" => "DB connection failed"], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

// ---- Read JSON or POST ----
$raw  = file_get_contents("php://input");
$data = json_decode($raw, true);
$isJson = is_array($data);

$mobile_money_number = trim($isJson ? ($data["mobile_money_number"] ?? "") : ($_POST["mobile_money_number"] ?? ""));
$policy_number       = trim($isJson ? ($data["policy_number"] ?? "") : ($_POST["policy_number"] ?? ""));
$full_name_deceased  = trim($isJson ? ($data["full_name_deceased"] ?? "") : ($_POST["full_name_deceased"] ?? ""));
$claim_type          = trim($isJson ? ($data["claim_type"] ?? "") : ($_POST["claim_type"] ?? ""));
$date_of_incident    = trim($isJson ? ($data["date_of_incident"] ?? "") : ($_POST["date_of_incident"] ?? ""));

// Optional: allow status from caller, else default
$claim_status        = trim($isJson ? ($data["claim_status"] ?? "SUBMITTED") : ($_POST["claim_status"] ?? "SUBMITTED"));

$userindex = ($mobile_money_number !== "") ? $mobile_money_number : "UNKNOWN";

// Log request received (input)
log_activity($con, $userindex, "REQUEST RECEIVED", $path, json_encode([
    "mobile_money_number" => $mobile_money_number,
    "policy_number" => $policy_number,
    "full_name_deceased" => $full_name_deceased,
    "claim_type" => $claim_type,
    "date_of_incident" => $date_of_incident,
    "claim_status" => $claim_status
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));

// ---- Validate minimal required ----
if ($mobile_money_number === "" || $policy_number === "" || $full_name_deceased === "" || $claim_type === "" || $date_of_incident === "") {
    out($con, $userindex, $path, false, ["error" => "Missing required fields"], "VALIDATION FAILED (missing fields)");
}

// ---- Validate incident date (YYYY-MM-DD) ----
$inc = DateTime::createFromFormat("Y-m-d", $date_of_incident);
$incErrors = DateTime::getLastErrors();

if (!$inc || ($incErrors["warning_count"] ?? 0) > 0 || ($incErrors["error_count"] ?? 0) > 0) {
    out($con, $userindex, $path, false, ["error" => "Invalid date_of_incident format. Use YYYY-MM-DD"], "VALIDATION FAILED (invalid date format)");
}

$today = new DateTime("today");
if ($inc > $today) {
    out($con, $userindex, $path, false, ["error" => "date_of_incident cannot be in the future"], "VALIDATION FAILED (future incident date)");
}

// Optional: normalize claim_status
$allowed_status = ["SUBMITTED","PENDING","APPROVED","DECLINED","PAID"];
$claim_status = strtoupper($claim_status);
if (!in_array($claim_status, $allowed_status, true)) {
    $claim_status = "SUBMITTED";
}

log_activity($con, $userindex, "STATUS NORMALIZED", $path, json_encode([
    "claim_status" => $claim_status
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));

/**
 * ✅ IMPORTANT CHANGE:
 * We REMOVED the duplicate check.
 * Customer can submit same policy + claim_type + date multiple times (duplicates allowed).
 * Each submission creates a NEW row with a NEW claim_id and claim_number.
 */

// ---- Generate claim number safely ----
$prefix = date("ymd"); // yymmdd (6 chars)
$lockName = "claims_seq_" . $prefix; // lock per day

log_activity($con, $userindex, "CLAIM NUMBER GENERATION START", $path, json_encode([
    "prefix" => $prefix,
    "lock" => $lockName
], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));

$gotLock = get_named_lock($con, $lockName, 8);
if (!$gotLock) {
    out($con, $userindex, $path, false, [
        "error" => "System busy. Please try again."
    ], "FAILED TO ACQUIRE LOCK");
}

mysqli_begin_transaction($con);

try {
    // Find the latest claim_number for today (under named lock => safe)
    $seqStmt = mysqli_prepare($con, "
        SELECT claim_number
        FROM subscription_claims
        WHERE claim_number LIKE CONCAT(?, '%')
        ORDER BY claim_id DESC
        LIMIT 1
    ");
    if (!$seqStmt) throw new Exception("seq prepare failed: " . mysqli_error($con));

    mysqli_stmt_bind_param($seqStmt, "s", $prefix);

    if (!mysqli_stmt_execute($seqStmt)) {
        $err = mysqli_stmt_error($seqStmt);
        mysqli_stmt_close($seqStmt);
        throw new Exception("seq execute failed: " . $err);
    }

    $seqRes = mysqli_stmt_get_result($seqStmt);
    mysqli_stmt_close($seqStmt);

    $nextSeq = 1;
    if ($seqRes && mysqli_num_rows($seqRes) > 0) {
        $last = (string)mysqli_fetch_assoc($seqRes)["claim_number"];
        $lastSeq = (int)substr($last, -5);
        $nextSeq = $lastSeq + 1;

        log_activity($con, $userindex, "LAST SEQUENCE FOUND", $path, json_encode([
            "last_claim_number" => $last,
            "last_seq" => $lastSeq,
            "next_seq" => $nextSeq
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
    } else {
        log_activity($con, $userindex, "NO SEQUENCE FOR TODAY", $path, json_encode([
            "next_seq" => $nextSeq
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));
    }

    $claim_number = $prefix . str_pad((string)$nextSeq, 5, "0", STR_PAD_LEFT);

    log_activity($con, $userindex, "CLAIM NUMBER GENERATED", $path, json_encode([
        "claim_number" => $claim_number
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));

    // ---- Insert claim (ALWAYS insert) ----
    $ins = mysqli_prepare($con, "
        INSERT INTO subscription_claims
        (mobile_money_number, claim_number, policy_number, full_name_deceased, claim_type, date_of_incident, claim_status, created_at)
        VALUES (?, ?, ?, ?, ?, ?, ?, NOW())
    ");
    if (!$ins) throw new Exception("insert prepare failed: " . mysqli_error($con));

    mysqli_stmt_bind_param(
        $ins,
        "sssssss",
        $mobile_money_number,
        $claim_number,
        $policy_number,
        $full_name_deceased,
        $claim_type,
        $date_of_incident,
        $claim_status
    );

    if (!mysqli_stmt_execute($ins)) {
        $err = mysqli_stmt_error($ins);
        mysqli_stmt_close($ins);
        throw new Exception("insert failed: " . $err);
    }

    mysqli_stmt_close($ins);

    $claim_id = (int)mysqli_insert_id($con);

    mysqli_commit($con);
    release_named_lock($con, $lockName);

    log_activity($con, $userindex, "CLAIM CREATED SUCCESSFULLY", $path, json_encode([
        "claim_id" => $claim_id,
        "claim_number" => $claim_number,
        "claim_status" => $claim_status
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));

    out($con, $userindex, $path, true, [
        "status" => "CREATED",
        "claim_id" => $claim_id,
        "claim_number" => $claim_number,
        "claim_status" => $claim_status
    ]);

} catch (Exception $e) {
    mysqli_rollback($con);
    release_named_lock($con, $lockName);

    log_activity($con, $userindex, "TRANSACTION FAILED", $path, json_encode([
        "error" => "failed",
        "details" => $e->getMessage()
    ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES));

    out($con, $userindex, $path, false, ["error" => "failed", "details" => $e->getMessage()]);
}
?>