<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1, maximum-scale=1, user-scalable=yes">
  <title>IMPACT LIFE · Video Training Library</title>
  <!-- Font Awesome 6 (free) for clean icons -->
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css">
  <style>
    /* ===== GLOBAL DESIGN TOKENS ===== */
    :root {
      --primary: #0a66c2;
      --primary-dark: #004182;
      --accent: #f9b81b;
      --accent-soft: #ffc940;
      --bg-dark: #0b0e14;
      --bg-panel: #151a22;
      --bg-card: #1e2630;
      --bg-hover: #2d3748;
      --text-light: #f1f5f9;
      --text-muted: #a0b3cc;
      --border-subtle: #2d3748;
      --shadow-lg: 0 20px 35px -10px rgba(0,0,0,0.6);
      --radius: 14px;
      --radius-sm: 8px;
      --font-main: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Helvetica, sans-serif;
    }

    * {
      margin: 0;
      padding: 0;
      box-sizing: border-box;
    }

    body {
      font-family: var(--font-main);
      background: linear-gradient(145deg, #0a0c10 0%, #030507 100%);
      color: var(--text-light);
      display: flex;
      height: 100vh;
      width: 100%;
      overflow: hidden;
      letter-spacing: 0.2px;
    }

    /* ===== PROFESSIONAL PLAYLIST SIDEBAR ===== */
    .playlist {
      width: 300px;
      background: var(--bg-panel);
      backdrop-filter: blur(4px);
      display: flex;
      flex-direction: column;
      border-right: 1px solid rgba(255,255,255,0.06);
      box-shadow: 8px 0 20px rgba(0,0,0,0.5);
      z-index: 10;
    }

    .playlist-header {
      padding: 24px 18px 16px;
      border-bottom: 1px solid var(--border-subtle);
    }

    .playlist-header h2 {
      font-size: 1.5rem;
      font-weight: 600;
      letter-spacing: -0.5px;
      display: flex;
      align-items: center;
      gap: 10px;
      color: white;
      margin: 0;
    }

    .playlist-header h2 i {
      color: var(--accent);
      font-size: 1.7rem;
    }

    .playlist-badge {
      background: rgba(249, 184, 27, 0.15);
      color: var(--accent-soft);
      font-size: 0.7rem;
      padding: 4px 10px;
      border-radius: 30px;
      font-weight: 500;
      display: inline-block;
      margin-top: 8px;
      border: 1px solid rgba(249,184,27,0.3);
    }

    .playlist-items-container {
      flex: 1;
      overflow-y: auto;
      padding: 16px 12px;
      display: flex;
      flex-direction: column;
      gap: 6px;
    }

    /* custom scroll */
    .playlist-items-container::-webkit-scrollbar {
      width: 5px;
    }
    .playlist-items-container::-webkit-scrollbar-track {
      background: transparent;
    }
    .playlist-items-container::-webkit-scrollbar-thumb {
      background: #3a4a5a;
      border-radius: 20px;
    }

    .playlist-item {
      display: flex;
      align-items: center;
      gap: 14px;
      padding: 14px 18px;
      background: var(--bg-card);
      border-radius: var(--radius-sm);
      cursor: pointer;
      transition: all 0.18s ease;
      border: 1px solid transparent;
      font-weight: 450;
      color: #e2e8f0;
    }

    .playlist-item i {
      font-size: 1.2rem;
      width: 24px;
      text-align: center;
      color: var(--accent);
      opacity: 0.8;
    }

    .playlist-item span {
      flex: 1;
      font-size: 0.95rem;
    }

    .playlist-item:hover {
      background: #26303f;
      border-color: #3e5063;
      transform: translateX(4px);
    }

    .playlist-item.active {
      background: linear-gradient(95deg, #1e3b5a, #182634);
      border-left: 6px solid var(--accent);
      border-radius: var(--radius-sm) 6px 6px var(--radius-sm);
      color: white;
      font-weight: 500;
      box-shadow: 0 6px 12px rgba(0,0,0,0.3);
    }

    .playlist-item.active i {
      color: white;
    }

    /* ===== VIDEO PLAYER AREA ===== */
    .video-player {
      flex: 1;
      display: flex;
      flex-direction: column;
      justify-content: center;
      align-items: center;
      padding: 30px 40px;
      background: radial-gradient(circle at 20% 30%, #141c24, #080b0e);
      position: relative;
    }

    .video-wrapper {
      width: 100%;
      max-width: 1000px;
      background: black;
      border-radius: var(--radius);
      box-shadow: var(--shadow-lg);
      overflow: hidden;
      border: 1px solid rgba(255,255,255,0.08);
    }

    video {
      width: 100%;
      height: auto;
      display: block;
      background: #000;
    }

    .video-controls-bar {
      display: flex;
      align-items: center;
      justify-content: space-between;
      width: 100%;
      max-width: 1000px;
      margin-top: 20px;
      padding: 6px 10px;
      background: rgba(10, 14, 23, 0.7);
      backdrop-filter: blur(12px);
      border-radius: 60px;
      border: 1px solid #334155;
    }

    .video-title {
      font-size: 0.95rem;
      display: flex;
      align-items: center;
      gap: 8px;
      color: #cbd5e1;
      font-weight: 500;
      padding-left: 10px;
    }

    .video-title i {
      color: var(--accent);
    }

    .btn {
      background: rgba(255,255,255,0.05);
      border: 1px solid #475569;
      color: white;
      padding: 10px 22px;
      border-radius: 40px;
      font-size: 0.9rem;
      font-weight: 500;
      display: flex;
      align-items: center;
      gap: 10px;
      transition: 0.15s;
      cursor: pointer;
      backdrop-filter: blur(5px);
    }

    .btn i {
      font-size: 1rem;
    }

    .btn:hover {
      background: var(--accent);
      border-color: var(--accent);
      color: black;
    }

    .btn:hover i {
      color: black;
    }

    /* empty state / small */
    .footer-note {
      position: absolute;
      bottom: 18px;
      right: 30px;
      font-size: 0.75rem;
      color: #5f7485;
    }

    /* responsive */
    @media (max-width: 700px) {
      body { flex-direction: column; }
      .playlist { width: 100%; height: 40vh; }
      .video-player { padding: 20px; }
    }
  </style>
</head>
<body>

<!-- =========== SIDEBAR: VIDEO LIBRARY =========== -->
<div class="playlist">
  <div class="playlist-header">
    <h2>
      <i class="fas fa-video"></i> IMPACT LIFE
    </h2>
    <span class="playlist-badge">
      <i class="fas fa-graduation-cap" style="margin-right: 6px;"></i>Abrabo Pa · User Manual
    </span>
  </div>

  <div class="playlist-items-container">
    <!-- playlist items – structured with unique IDs, proper video/sub sources -->
    <div class="playlist-item active" data-id="vid_sysadmin" data-src="videos/sys_admin_settings.mp4" data-sub="subs/sys_admin.vtt">
      <i class="fas fa-shield-alt"></i>
      <span>System Administrator</span>
      <span style="font-size: 0.7rem; opacity: 0.7; margin-left: auto;"><i class="fas fa-clock"></i> 4:20</span>
    </div>
    <div class="playlist-item" data-id="vid_telesales" data-src="videos/telesales_agents.mp4" data-sub="subs/telesales.vtt">
      <i class="fas fa-headset"></i>
      <span>Telesales Agents</span>
    </div>
    <div class="playlist-item" data-id="vid_backoffice" data-src="videos/backoffice.mp4" data-sub="subs/backoffice.vtt">
      <i class="fas fa-building"></i>
      <span>Back Office</span>
    </div>
    <div class="playlist-item" data-id="vid_premium" data-src="videos/premium_allocation.mp4" data-sub="subs/premium_allocation.vtt">
      <i class="fas fa-coins"></i>
      <span>Premium Admin</span>
    </div>
    <div class="playlist-item" data-id="vid_claimsofficer" data-src="videos/claims_office.mp4" data-sub="subs/claims_officer.vtt">
      <i class="fas fa-file-invoice"></i>
      <span>Claims Officer</span>
    </div>
    <div class="playlist-item" data-id="vid_claimsmanager" data-src="videos/claims_manager.mp4" data-sub="subs/claims_manager.vtt">
      <i class="fas fa-user-tie"></i>
      <span>Claims Manager</span>
    </div>
    <div class="playlist-item" data-id="vid_finance" data-src="videos/finance_manager.mp4" data-sub="subs/finance_manager.vtt">
      <i class="fas fa-chart-line"></i>
      <span>Finance Manager</span>
    </div>
  </div>
  
  <!-- subtle extra info (training) -->
  <div style="padding: 16px 18px; border-top: 1px solid #263238; font-size: 0.8rem; color: #99aabb;">
    <i class="fas fa-subtitle" style="margin-right: 6px;"></i> Captions ready • Resume enabled
  </div>
</div>

<!-- =========== VIDEO PLAYER + CAPTION TOGGLE =========== -->
<div class="video-player">
  <div class="video-wrapper">
    <video id="mainVideo" controls preload="metadata">
      <!-- default source - sys admin -->
      <source src="videos/sys_admin_settings.mp4" type="video/mp4">
      <!-- default subtitle track - dynamically replaced -->
      <track id="captionTrack" src="subs/sys_admin.vtt" kind="subtitles" srclang="en" label="English" default>
      Your browser does not support the video tag.
    </video>
  </div>
  
  <!-- sleek control bar -->
  <div class="video-controls-bar">
    <span class="video-title">
      <i class="fas fa-play-circle"></i> 
      <span id="currentVideoLabel">System Administrator</span>
    </span>
    <button id="toggleCaptionsBtn" class="btn">
      <i class="fas fa-closed-captioning"></i> 
      <span id="captionBtnText">Captions ON</span>
    </button>
  </div>
  <div class="footer-note">
    <i class="fas fa-phone-alt"></i> Abrabo Pa · v1.2
  </div>
</div>

<script>
  (function() {
    "use strict";

    // ----- DOM elements -----
    const video = document.getElementById('mainVideo');
    const captionTrack = document.getElementById('captionTrack');
    const toggleBtn = document.getElementById('toggleCaptionsBtn');
    const captionBtnText = document.getElementById('captionBtnText');
    const currentVideoLabel = document.getElementById('currentVideoLabel');

    // ----- State -----
    let currentVideoId = 'vid_sysadmin';         // unique identifier
    let currentSubPath = 'subs/sys_admin.vtt';   // fallback

    // ----- HELPER: save / load playback position -----
    function savePlaybackTime(videoId, time) {
      if (videoId && typeof time === 'number' && isFinite(time)) {
        localStorage.setItem(`impact_playback_${videoId}`, time.toString());
      }
    }

    function loadPlaybackTime(videoId) {
      const saved = localStorage.getItem(`impact_playback_${videoId}`);
      return saved ? parseFloat(saved) : 0;
    }

    // ----- Update active state in playlist (visual) -----
    function setActivePlaylistItem(targetId) {
      document.querySelectorAll('.playlist-item').forEach(el => {
        el.classList.remove('active');
      });
      const activeItem = document.querySelector(`.playlist-item[data-id="${targetId}"]`);
      if (activeItem) {
        activeItem.classList.add('active');
        // update label
        const spanText = activeItem.querySelector('span:first-of-type')?.innerText || activeItem.innerText.split('📽️').pop() || 'Video';
        currentVideoLabel.textContent = spanText.replace(/[📽️•]/g, '').trim() || 'Training';
      }
    }

    // ----- Update video source and subtitle -----
    function loadVideo(videoSrc, subtitleSrc, videoId, videoTitleText) {
      if (!videoSrc) return;

      // Save current position for previous video before switching
      if (currentVideoId) {
        savePlaybackTime(currentVideoId, video.currentTime);
      }

      // Update source
      const sourceEl = video.querySelector('source');
      if (sourceEl) {
        sourceEl.src = videoSrc;
      } else {
        // fallback: create new source
        const newSource = document.createElement('source');
        newSource.src = videoSrc;
        newSource.type = 'video/mp4';
        video.appendChild(newSource);
      }

      // Update subtitle track
      if (captionTrack) {
        captionTrack.src = subtitleSrc || '';
      }

      // refresh & load
      video.load();
      currentVideoId = videoId;
      currentSubPath = subtitleSrc;

      // restore playback position for new video
      const savedTime = loadPlaybackTime(currentVideoId);
      
      // important: wait until video metadata is ready to set currentTime
      const setTimeHandler = () => {
        if (savedTime > 0.5) { // only seek if meaningful
          video.currentTime = savedTime;
        }
        video.removeEventListener('loadedmetadata', setTimeHandler);
      };
      video.addEventListener('loadedmetadata', setTimeHandler);
      
      // fallback if metadata already loaded
      if (video.readyState >= 1) {
        if (savedTime > 0.5) video.currentTime = savedTime;
      }

      // update active visual
      setActivePlaylistItem(videoId);
    }

    // ----- CAPTIONS TOGGLE (professional) -----
    function updateCaptionButtonState() {
      const tracks = video.textTracks;
      let isShowing = false;
      for (let i = 0; i < tracks.length; i++) {
        if (tracks[i].kind === 'subtitles' || tracks[i].kind === 'captions') {
          if (tracks[i].mode === 'showing') {
            isShowing = true;
          }
          break; // we assume only one track
        }
      }
      if (isShowing) {
        captionBtnText.innerText = 'Captions ON';
        toggleBtn.classList.add('btn-active'); // optional
      } else {
        captionBtnText.innerText = 'Captions OFF';
        toggleBtn.classList.remove('btn-active');
      }
    }

    // Toggle caption visibility
    function toggleCaptions() {
      const tracks = video.textTracks;
      for (let i = 0; i < tracks.length; i++) {
        if (tracks[i].kind === 'subtitles' || tracks[i].kind === 'captions') {
          tracks[i].mode = (tracks[i].mode === 'showing') ? 'hidden' : 'showing';
        }
      }
      updateCaptionButtonState();
    }

    // ----- Initialize event listeners -----
    function initEventListeners() {
      // 1. Save playback progress every 2 seconds (throttled)
      let saveTimeout;
      video.addEventListener('timeupdate', () => {
        if (!currentVideoId) return;
        if (saveTimeout) clearTimeout(saveTimeout);
        saveTimeout = setTimeout(() => {
          savePlaybackTime(currentVideoId, video.currentTime);
        }, 400); // smooth
      });

      // 2. Toggle captions button
      toggleBtn.addEventListener('click', toggleCaptions);

      // 3. Playlist item selection
      document.querySelectorAll('.playlist-item').forEach(item => {
        item.addEventListener('click', (e) => {
          const src = item.dataset.src;
          const sub = item.dataset.sub;
          const id = item.dataset.id;
          if (src && id) {
            loadVideo(src, sub, id);
          }
        });
      });

      // 4. When video metadata loads, ensure button reflects correct caption state
      video.addEventListener('loadedmetadata', () => {
        updateCaptionButtonState();
      });

      // 5. optional: detect caption track changes
      video.addEventListener('cuechange', updateCaptionButtonState);
    }

    // ----- initial restore & caption sync -----
    // set default active video label
    setActivePlaylistItem('vid_sysadmin');
    
    // set initial caption button state (wait for metadata)
    if (video.readyState >= 1) {
      updateCaptionButtonState();
    } else {
      video.addEventListener('loadedmetadata', updateCaptionButtonState, { once: true });
    }

    // restore last position for default video
    const defaultSaved = loadPlaybackTime('vid_sysadmin');
    if (defaultSaved > 0.5 && video.readyState >= 1) {
      video.currentTime = defaultSaved;
    } else if (defaultSaved > 0.5) {
      video.addEventListener('loadedmetadata', () => {
        video.currentTime = defaultSaved;
      }, { once: true });
    }

    // start everything
    initEventListeners();

    // Ensure caption track is loaded properly: sometimes default not showing
    // Force track initial mode to 'showing' for default captions
    video.addEventListener('loadstart', () => {
      const tracks = video.textTracks;
      for (let i = 0; i < tracks.length; i++) {
        if (tracks[i].kind === 'subtitles' && tracks[i].label === 'English') {
          // we want captions ON by default for accessibility
          tracks[i].mode = 'showing';
        }
      }
      updateCaptionButtonState();
    }, { once: true });

  })();
</script>

<!-- ensure captions default to showing on first load -->
<noscript>JavaScript recommended for caption toggle & resume.</noscript>
</body>
</html>