<?php
/**
 * sendHubtelSms()
 * --------------------------------------------------
 * Sends SMS using Hubtel GET endpoint
 * Logs every attempt into table: sms_logs
 *
 * TABLE: sms_logs
 *  sms_log_id
 *  user_id
 *  mobile_number
 *  formatted_number
 *  message_type
 *  http_code
 *  response
 *  created_at
 *
 * @param mysqli $con
 * @param int    $user_id
 * @param string $mobile_number  Original number
 * @param string $content        SMS content
 * @param string $message_type   e.g. TNC, OTP, ALERT
 * @param string $from           Sender ID
 *
 * @return array
 */

function sendHubtelSms(
    mysqli $con,
    int $user_id,
    string $mobile_number,
    string $content,
    string $message_type = "GENERAL",
    string $from = "ImpactLife"
) {
    // =======================
    // Hubtel credentials
    // =======================
    $clientId     = "sdjzxukb";
    $clientSecret = "bhbevesa";

    // =======================
    // Normalize number
    // =======================
    $formatted_number = preg_match('/^233\d{9}$/', $mobile_number)
        ? $mobile_number
        : (preg_match('/^0\d{9}$/', $mobile_number)
            ? "233" . substr($mobile_number, 1)
            : $mobile_number);

    // =======================
    // Build API URL
    // =======================
    $apiUrl = "https://sms.hubtel.com/v1/messages/send?" . http_build_query([
        "clientsecret" => $clientSecret,
        "clientid"     => $clientId,
        "from"         => $from,
        "to"           => $formatted_number,
        "content"      => $content
    ]);

    // =======================
    // cURL Request
    // =======================
    $ch = curl_init();
    curl_setopt_array($ch, [
        CURLOPT_URL            => $apiUrl,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT        => 30,
        CURLOPT_CUSTOMREQUEST  => "GET",
        CURLOPT_HTTPHEADER     => [
            "Accept: application/json"
        ]
    ]);

    $response  = curl_exec($ch);
    $httpCode  = (int) curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $curlError = curl_error($ch);
    curl_close($ch);

    if ($response === false || $response === null) {
        $response = $curlError;
    }

    // =======================
    // Log to sms_logs
    // =======================
    $stmt = $con->prepare("
        INSERT INTO sms_logs
        (user_id, mobile_number, formatted_number, message_type, http_code, response, created_at)
        VALUES (?, ?, ?, ?, ?, ?, NOW())
    ");

    if ($stmt) {
        $stmt->bind_param(
            "isssis",
            $user_id,
            $mobile_number,
            $formatted_number,
            $message_type,
            $httpCode,
            $response
        );
        $stmt->execute();
        $stmt->close();
    }

    // =======================
    // Return result
    // =======================
    return [
        "success"          => ($curlError === "" && $httpCode >= 200 && $httpCode < 300),
        "http_code"        => $httpCode,
        "curl_error"       => $curlError,
        "response"         => $response,
        "formatted_number" => $formatted_number
    ];
}
