<?php
/**
 * Abrabo Pa – Claims Pay Function (Uses policy_rules_view + cover_levels_view)
 * PHP 7.x / XAMPP SAFE (no get_result)
 */

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

date_default_timezone_set("Africa/Accra");

/**
 * Main function you will call
 */
function abraboPayClaim(mysqli $con, array $in): array
{
    // ----------------- Helpers -----------------
    $toTs = function($d){
        if ($d === null) return null;
        $d = trim((string)$d);
        if ($d === "") return null;
        $ts = strtotime($d);
        return $ts === false ? null : $ts;
    };

    $addMonths = function(int $baseTs, int $months): int {
        return strtotime("+{$months} months", $baseTs);
    };

    // XAMPP-safe: bind_result instead of get_result
    $getRuleMonths = function(mysqli $con, string $lifeTypeName, int $age, string $ruleCode): ?int {
        $sql = "
            SELECT waiting_period_months
            FROM policy_rules_view
            WHERE life_type = ?
              AND rule_code = ?
              AND ? BETWEEN min_age AND max_age
            ORDER BY waiting_period_months DESC
            LIMIT 1
        ";
        $stmt = $con->prepare($sql);
        if (!$stmt) return null;

        $stmt->bind_param("ssi", $lifeTypeName, $ruleCode, $age);
        $stmt->execute();

        $waitingMonths = null;
        $stmt->bind_result($waitingMonths);
        $ok = $stmt->fetch();
        $stmt->close();

        if ($ok) return (int)$waitingMonths;
        return null;
    };

    $getSumAssuredByCoverId = function(mysqli $con, int $coverLevelId): ?float {
        $sql = "
            SELECT sum_assured
            FROM cover_levels_view
            WHERE cover_level_id = ?
              AND is_active = 1
            LIMIT 1
        ";
        $stmt = $con->prepare($sql);
        if (!$stmt) return null;

        $stmt->bind_param("i", $coverLevelId);
        $stmt->execute();

        $sumAssured = null;
        $stmt->bind_result($sumAssured);
        $ok = $stmt->fetch();
        $stmt->close();

        if ($ok) return (float)$sumAssured;
        return null;
    };

    // ----------------- Output -----------------
    $out = [
        "ok" => false,
        "payable_amount" => 0.00,
        "effective_sum_assured" => 0.00,
        "waiting_months" => 0,
        "waiting_end_date" => null,
        "upgrade_allowed" => null,
        "upgrade_wait_end_date" => null,
        "reason" => "",
        "debug" => []
    ];

    // ----------------- Required inputs -----------------
    $life_type  = strtoupper(trim($in["life_type"] ?? "")); // MAIN / SECONDARY / ADDED_SECONDARY
    $claim_type = strtoupper(trim($in["claim_type"] ?? "")); // DEATH / ACCIDENTAL_DEATH / CRITICAL_ILLNESS
    $age        = (int)($in["life_age"] ?? 0);

    $event_ts      = $toTs($in["event_date"] ?? null);
    $notify_ts     = $toTs($in["notification_date"] ?? null);
    $inception_ts  = $toTs($in["inception_date"] ?? null);
    $reinst_ts     = $toTs($in["last_reinstatement_date"] ?? null);
    $added_sec_ts  = $toTs($in["secondary_added_date"] ?? null);

    $current_cover_level_id  = (int)($in["current_cover_level_id"] ?? 0);
    $previous_cover_level_id = (int)($in["previous_cover_level_id"] ?? 0);
    $upgrade_ts              = $toTs($in["cover_upgrade_date"] ?? null);

    if (!$life_type || !$claim_type || $age <= 0 || !$event_ts || !$notify_ts || !$inception_ts || $current_cover_level_id <= 0) {
        $out["reason"] = "Missing required inputs: life_type, claim_type, life_age, event_date, notification_date, inception_date, current_cover_level_id.";
        return $out;
    }

    // ----------------- Gate checks -----------------
    $is_policy_active = (bool)($in["is_policy_active"] ?? true);
    $is_excluded      = (bool)($in["is_excluded"] ?? false);
    $has_misrep       = (bool)($in["has_misrepresentation"] ?? false);

    if (!$is_policy_active) { $out["reason"] = "Policy not active (lapsed) at event date."; return $out; }
    if ($is_excluded)       { $out["reason"] = "Claim excluded by policy exclusions."; return $out; }
    if ($has_misrep)        { $out["reason"] = "Misrepresentation / non-disclosure flagged."; return $out; }

    // Notification within 3 months
    if ($notify_ts > strtotime("+3 months", $event_ts)) {
        $out["reason"] = "Claim notification beyond 3 months from event date.";
        return $out;
    }

    // ----------------- Map life type -----------------
    $baseLifeName = "";
    if ($life_type === "MAIN") $baseLifeName = "Main Life";
    elseif ($life_type === "SECONDARY") $baseLifeName = "Secondary Life";
    elseif ($life_type === "ADDED_SECONDARY") $baseLifeName = "Added Secondary Life";
    else {
        $out["reason"] = "Invalid life_type. Use MAIN / SECONDARY / ADDED_SECONDARY.";
        return $out;
    }

    $rulesLifeName = $baseLifeName;
    if ($reinst_ts) {
        if ($life_type === "MAIN") $rulesLifeName = "Main Life (Reinstatement)";
        if ($life_type === "SECONDARY") $rulesLifeName = "Secondary Life (Reinstatement)";
    }

    // ----------------- Waiting period -----------------
    $is_accidental = ($claim_type === "ACCIDENTAL_DEATH");
    $waitingMonths = 0;

    if (!$is_accidental) {
        if ($life_type === "ADDED_SECONDARY" || ($life_type === "SECONDARY" && $added_sec_ts)) {
            $wm = $getRuleMonths($con, "Added Secondary Life", $age, "WAITING_PERIOD_ADDED_SECONDARY");
            if ($wm === null) { $out["reason"] = "Missing WAITING_PERIOD_ADDED_SECONDARY rule in policy_rules_view."; return $out; }
            $waitingMonths = $wm;
        } else {
            $wm = $getRuleMonths($con, $rulesLifeName, $age, "WAITING_PERIOD_GENERAL");
            if ($wm === null) {
                $wm2 = $getRuleMonths($con, $rulesLifeName, $age, "REINSTATEMENT");
                if ($wm2 === null) { $out["reason"] = "Missing waiting rule (WAITING_PERIOD_GENERAL/REINSTATEMENT) in policy_rules_view."; return $out; }
                $waitingMonths = $wm2;
            } else {
                $waitingMonths = $wm;
            }
        }
    }

    $out["waiting_months"] = $waitingMonths;

    $waitingBase = $inception_ts;
    if ($reinst_ts && $reinst_ts > $waitingBase) $waitingBase = $reinst_ts;
    if ($added_sec_ts && $added_sec_ts > $waitingBase) $waitingBase = $added_sec_ts;

    $waitingEnd = $addMonths($waitingBase, $waitingMonths);
    $out["waiting_end_date"] = date("Y-m-d", $waitingEnd);

    if (!$is_accidental && $event_ts < $waitingEnd) {
        $out["reason"] = "Event occurred within waiting period (non-accidental).";
        $out["debug"]["waiting_base"] = date("Y-m-d", $waitingBase);
        return $out;
    }

    // ----------------- Sum assured -----------------
    $newSA = $getSumAssuredByCoverId($con, $current_cover_level_id);
    if ($newSA === null) { $out["reason"] = "Cannot fetch sum_assured for current_cover_level_id from cover_levels_view."; return $out; }

    $oldSA = $newSA;
    if ($previous_cover_level_id > 0) {
        $tmp = $getSumAssuredByCoverId($con, $previous_cover_level_id);
        if ($tmp === null) { $out["reason"] = "Cannot fetch sum_assured for previous_cover_level_id from cover_levels_view."; return $out; }
        $oldSA = $tmp;
    }

    // ----------------- Cover upgrade rule -----------------
    $effectiveSA = $newSA;

    if ($upgrade_ts) {
        $sixMonthsAfterInception = $addMonths($inception_ts, 6);

        if ($upgrade_ts < $sixMonthsAfterInception) {
            $out["upgrade_allowed"] = false;
            $effectiveSA = $oldSA; // ignore upgrade
        } else {
            $out["upgrade_allowed"] = true;

            $upgradeMonths = $getRuleMonths($con, $baseLifeName, $age, "COVER_CHANGE_WAITING");
            if ($upgradeMonths === null) $upgradeMonths = 0;

            if ($upgradeMonths > 0) {
                $upgradeWaitEnd = $addMonths($upgrade_ts, $upgradeMonths);
                $out["upgrade_wait_end_date"] = date("Y-m-d", $upgradeWaitEnd);

                $effectiveSA = ($event_ts < $upgradeWaitEnd) ? $oldSA : $newSA;
            } else {
                $effectiveSA = $newSA;
            }
        }
    }

    $out["effective_sum_assured"] = round((float)$effectiveSA, 2);

    // ----------------- Payable formulas -----------------
    $ci_paid_amount = (float)($in["ci_paid_amount"] ?? 0.0);

    if ($claim_type === "CRITICAL_ILLNESS") {
        if ($life_type !== "MAIN") {
            $out["reason"] = "Critical illness benefit applies to MAIN only.";
            return $out;
        }

        $allowedCI = ["HEART_ATTACK","STROKE","CANCER","CAD_SURGERY","KIDNEY_FAILURE","MAJOR_ORGAN_TRANSPLANT","MAJOR_BURNS"];
        $ciType = strtoupper(trim($in["critical_illness_type"] ?? ""));
        if ($ciType !== "" && !in_array($ciType, $allowedCI, true)) {
            $out["reason"] = "Critical illness type not eligible.";
            return $out;
        }

        $out["payable_amount"] = round($effectiveSA * 0.5, 2);
        $out["ok"] = true;
        $out["reason"] = "Approved: Critical illness pays 50% of Sum Assured.";
        return $out;
    }

    if ($claim_type === "DEATH" || $claim_type === "ACCIDENTAL_DEATH") {
        if ($life_type === "MAIN") {
            $out["payable_amount"] = round(max(0, $effectiveSA - $ci_paid_amount), 2);
            $out["ok"] = true;
            $out["reason"] = "Approved: Death pays Sum Assured minus any CI already paid.";
            return $out;
        }

        $out["payable_amount"] = round($effectiveSA, 2);
        $out["ok"] = true;
        $out["reason"] = "Approved: Death pays full Sum Assured for Secondary/Added Secondary.";
        return $out;
    }

    $out["reason"] = "Invalid claim_type. Use DEATH / ACCIDENTAL_DEATH / CRITICAL_ILLNESS.";
    return $out;
}

/* ===========================
   TEST / EXAMPLE CALL
   =========================== 

$calc = abraboPayClaim($con, [
    "life_type" => "MAIN",
    "life_age" => 45,
    "claim_type" => "DEATH",
    "event_date" => "2026-01-10",
    "notification_date" => "2026-01-20",
    "inception_date" => "2025-06-01",
    "last_reinstatement_date" => null,
    "secondary_added_date" => null,

    "is_policy_active" => true,
    "is_excluded" => false,
    "has_misrepresentation" => false,

    "current_cover_level_id" => 5,         // 10,000
    "previous_cover_level_id" => 4,        // 8,000
    "cover_upgrade_date" => "2025-12-15",

    "ci_paid_amount" => 5000,
    "critical_illness_type" => "STROKE",
]);

echo "<pre>";
print_r($calc);
echo "</pre>";

if ($calc["ok"]) {
    echo "<b>APPROVED AMOUNT:</b> " . $calc["payable_amount"];
} else {
    echo "<b>DECLINED REASON:</b> " . htmlspecialchars($calc["reason"]);
}
*/