<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

include("database.php");
require_once('generate_claim_id.php');
require_once('rules_calculation.php');   // must contain abraboPayClaim()
require_once('laspse_function.php');     // if you want to compute active/lapsed
include('log_functions.php');  
include("session.php");

date_default_timezone_set("Africa/Accra");

// Validate session user
$userindex = isset($_SESSION['userindex']) ? (int)$_SESSION['userindex'] : 0;

// Helper: safer request getter
function req(mysqli $con, string $key): string {
    $val = $_REQUEST[$key] ?? '';
    $val = is_string($val) ? $val : '';
    $val = mysqli_real_escape_string($con, $val);
    return trim($val);
}

// Inputs
$policy_number = req($con, 'policy_number');
$policyholder_id = req($con, 'policyholder_id');
$life_type = strtoupper(req($con, 'life_type')); // MAIN/SECONDARY/BENEFICIARY
$life_selected_id = req($con, 'life_selected_id');
$claim_type = strtoupper(req($con, 'claim_type'));
$critical_illness_type = strtoupper(req($con, 'critical_illness_type'));

$date_of_claim_notification = req($con, 'date_of_claim_notification');
$claim_dateofevent = req($con, 'claim_dateofevent');
$claim_causeofevent = req($con, 'claim_causeofevent');
$claim_dateofstatement = req($con, 'claim_dateofstatement'); // ✅ fixed
$claim_status = req($con, 'claim_status');
$remarks = req($con, 'remarks');

$payout_momo = req($con, 'payout_momo');
$payout_reference = req($con, 'payout_reference');

$inception_date = req($con, 'inception_date');
$cover_level_id = (int)req($con, 'cover_level_id');

$age_posted = (int)req($con, 'age'); // keep posted age for MAIN

// Map BENEFICIARY (your UI) to something your engine supports, or block it
// OPTION 1: Block (recommended unless you have beneficiary rules)
if ($life_type === 'BENEFICIARY') {
    die("Life type BENEFICIARY is not supported by abraboPayClaim(). Use MAIN/SECONDARY/ADDED_SECONDARY.");
}


//echo $life_selected_id;exit;


// Determine age correctly
$age = $age_posted; // default for MAIN
if ($life_type === 'SECONDARY') {
    // Age from secondary_lives table
    $age = 0;
    $stmt = $con->prepare("
        SELECT age_at_entry
        FROM secondary_lives
        WHERE secondary_life_id = ?
        LIMIT 1
    ");
    if (!$stmt) die("Prepare failed: " . $con->error);

    $secId = (int)$life_selected_id;
    $stmt->bind_param("i", $secId);
    $stmt->execute();
    $result = $stmt->get_result();
    if (!$result || $result->num_rows === 0) die("Secondary life not found.");

    $row = $result->fetch_assoc();
    $age = (int)($row['age_at_entry'] ?? 0);
    $stmt->close();
}



// Optional: compute policy active today (recommended)
$chk = abraboIsPolicyLapsed($con, $policy_number, date('Y-m-d'));
$is_policy_active = ($chk['ok'] ?? false) && !($chk['is_lapsed'] ?? true);

// Initialize outputs to avoid undefined variables
$payable_amount = 0.0;
$effective_sum_assured = 0.0;
$decline_reason = "";


$data_inf =
    "policy_number: {$policy_number}\n" .
    "policyholder_id: {$policyholder_id}\n" .
    "life_type: {$life_type}\n" .
    "life_selected_id: {$life_selected_id}\n" .
    "claim_type: {$claim_type}\n" .
    "critical_illness_type: {$critical_illness_type}\n\n" .

    "date_of_claim_notification: {$date_of_claim_notification}\n" .
    "claim_dateofevent: {$claim_dateofevent}\n" .
    "claim_causeofevent: {$claim_causeofevent}\n" .
    "claim_dateofstatement: {$claim_dateofstatement}\n" .
    "claim_status: {$claim_status}\n" .
    "remarks: {$remarks}\n\n" .

    "payout_momo: {$payout_momo}\n" .
    "payout_reference: {$payout_reference}\n\n" .

    "inception_date: {$inception_date}\n" .
    "cover_level_id: {$cover_level_id}\n" .
    "age_posted: {$age}";




//Run claim engine
$calc = abraboPayClaim($con, [
    "life_type" => $life_type,
    "life_age" => $age,
    "claim_type" => $claim_type,
    "event_date" => $claim_dateofevent,
    "notification_date" => $date_of_claim_notification,
    "inception_date" => $inception_date,

    "last_reinstatement_date" => null,
    "secondary_added_date" => null,

    "is_policy_active" => $is_policy_active, // ✅ not hardcoded
    "is_excluded" => false,
    "has_misrepresentation" => false,

    "current_cover_level_id" => $cover_level_id,
    "previous_cover_level_id" => 0,
    "cover_upgrade_date" => null,

    "ci_paid_amount" => 0,
    "critical_illness_type" => $critical_illness_type,
]);

if (!empty($calc["ok"])) {
    $payable_amount        = (float)($calc['payable_amount'] ?? 0);
    $effective_sum_assured = (float)($calc['effective_sum_assured'] ?? 0);
} else {
    $decline_reason = (string)($calc['reason'] ?? 'Declined');
}


// Debug output
//echo "Claim ID: {$claim_id} | Payable: {$payable_amount} | Sum Assured: {$effective_sum_assured}";


if (!empty($decline_reason)) {
    $msg = "DECLINED: " . $decline_reason;
    echo "<script>
        alert(" . json_encode($msg) . ");
        window.history.back();
    </script>";
    exit;
}





$con->begin_transaction();

try {

     $claims_date = date('Y-m-d');

    // Generate Claim id
    $claim_id = generateClaimId($con);

        // If you want claims_date to be the statement date or paid date, set it here
    // Example:
    // $claims_date = $claim_dateofstatement; // or null

    // ===================================
    // 1. INSERT INTO claims
    // ===================================
    $stmt = $con->prepare("
        INSERT INTO claims (
            claim_id,
            policy_number,
            policyholder_id,
            life_type,
            secondary_life_id,
            claim_type,
            critical_illness_type,
            event_date,
            reported_date,
            sum_assured,
            approved_amount,
            claim_status,
            decision_notes,
            payout_momo,
            payout_reference,
            claims_date,
            userindex,
            userinput_date
        ) VALUES (
            ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, NOW()
        )
    ");
    if (!$stmt) {
        throw new Exception('Prepare failed (claims): ' . $con->error);
    }

    // Types:
    // claim_id(s) policy_number(s) policyholder_id(s) life_type(s)
    // secondary_life_id(i) claim_type(s) critical_illness_type(s)
    // event_date(s) reported_date(s)
    // sum_assured(d) approved_amount(d)
    // claim_status(s) decision_notes(s)
    // payout_momo(s) payout_reference(s)
    // claims_date(s) userindex(i)
    $stmt->bind_param(
        "ssssissssddsssssi",
        $claim_id,
        $policy_number,
        $policyholder_id,
        $life_type,
        $life_selected_id,
        $claim_type,
        $critical_illness_type,
        $claim_dateofevent,
        $date_of_claim_notification,
        $effective_sum_assured,
        $payable_amount,
        $claim_status,
        $remarks,
        $payout_momo,
        $payout_reference,
        $claims_date,
        $userindex
    );

    if (!$stmt->execute()) {
        throw new Exception("Insert claim failed: " . $stmt->error);
    }
    $stmt->close();

    // ===================================
    // 2. UPLOAD + INSERT DOCUMENTS
    // ===================================
    if (isset($_FILES['claim_documents']) && !empty($_FILES['claim_documents']['name'][0])) {

        $uploadDir = "uploads/" . $claim_id . "/";

        if (!is_dir($uploadDir) && !mkdir($uploadDir, 0777, true)) {
            throw new Exception("Failed to create upload folder");
        }

        $allowed = ['pdf', 'jpg', 'jpeg', 'png'];

        foreach ($_FILES['claim_documents']['name'] as $key => $filename) {

            $tmpName = $_FILES['claim_documents']['tmp_name'][$key] ?? '';
            $error   = $_FILES['claim_documents']['error'][$key] ?? UPLOAD_ERR_NO_FILE;

            if ($error !== UPLOAD_ERR_OK) continue;

            $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
            if (!in_array($ext, $allowed, true)) continue;

            $safeName = preg_replace("/[^a-zA-Z0-9._-]/", "_", $filename);
            $newName  = time() . "_" . bin2hex(random_bytes(4)) . "_" . $safeName;
            $filePath = $uploadDir . $newName;

            if (!move_uploaded_file($tmpName, $filePath)) {
                throw new Exception("Failed to move file: $filename");
            }

            // IMPORTANT: use the real column name in your table:
            // uploaded_by (NOT userindex) unless your table is different
            $stmt = $con->prepare("
                INSERT INTO claim_documents (
                    claim_id,
                    doc_type,
                    file_path,
                    uploaded_at,
                    userindex
                ) VALUES (
                    ?, ?, ?, NOW(), ?
                )
            ");
            if (!$stmt) {
                throw new Exception("Prepare failed (documents): " . $con->error);
            }

            $doc_type = strtoupper($ext);

            $stmt->bind_param("sssi", $claim_id, $doc_type, $filePath, $userindex);

            if (!$stmt->execute()) {
                throw new Exception("Insert document failed: " . $stmt->error);
            }
            $stmt->close();
        }
    }

    $con->commit();


     // Log the successful operation
                $page_path = 'policy_claims_initated_eng.php';
                $reasons = 'Claim successfully submitted. Claim ID Generated';

 //$data_inf ='N/A';

                log_failed_login($con, $userindex, $page_path, $reasons ,$data_inf);

    echo "<script>
        alert('Claim successfully submitted. Claim ID Generated: " . addslashes($claim_id) . "');
        window.location.href = 'policy_claims_search.php';
    </script>";
    exit;

} catch (Exception $e) {
    $con->rollback();
    echo "<script>
        alert(" . json_encode("Error: " . $e->getMessage()) . ");
        window.history.back();
    </script>";
    exit;
}
