<?php
// generate_claim_id.php
// =====================================================
// Generates Claim ID like: CL01260000001
// Rules (your requirement):
// - Uses table: generate_idn
// - Always uses the single row where generateindex = 1
// - Every call increments the last 5-digit sequence by +1 (same month)
// - If month changes, reset sequence to 00001 and update YYMM
// - PHP 7.x compatible (no str_starts_with)
// =====================================================

/**
 * Generate and update claim_id in generate_idn where generateindex = 1
 * Returns the new claim_id (string).
 */
function generateClaimId(mysqli $con): string
{
    $prefix = "CL";
    $currentYYMM = date("ym"); // e.g. 0126 (Jan 2026)

    $con->begin_transaction();

    try {
        // Lock row to prevent duplicates (multi-user safe)
        $stmt = $con->prepare("
            SELECT claim_id
            FROM generate_idn
            WHERE generateindex = 1
            FOR UPDATE
        ");
        if (!$stmt) {
            throw new Exception("Prepare failed: " . $con->error);
        }

        $stmt->execute();
        $res = $stmt->get_result();

        if (!$res || $res->num_rows === 0) {
            throw new Exception("Row generateindex=1 not found in generate_idn.");
        }

        $row = $res->fetch_assoc();
        $lastClaimId = trim((string)($row['claim_id'] ?? ''));

        // Default
        $nextNumber = 1;

        // Validate & parse old claim id (expecting CL + YYMM + 5 digits)
        // Example: CL01260000001
        if (
            $lastClaimId !== '' &&
            strlen($lastClaimId) >= 11 &&
            substr($lastClaimId, 0, strlen($prefix)) === $prefix
        ) {
            $lastYYMM = substr($lastClaimId, 2, 4); // YYMM
            $lastSeq  = substr($lastClaimId, -5);   // last 5 digits

            if ($lastYYMM === $currentYYMM && ctype_digit($lastSeq)) {
                // Same month -> increment
                $nextNumber = ((int)$lastSeq) + 1;
            } else {
                // New month (or invalid old data) -> reset
                $nextNumber = 1;
            }
        }

        // Build new claim id
        $seq = str_pad((string)$nextNumber, 5, "0", STR_PAD_LEFT);
        $newClaimId = $prefix . $currentYYMM . $seq;

        // Update the single row
        $upd = $con->prepare("
            UPDATE generate_idn
            SET claim_id = ?
            WHERE generateindex = 1
        ");
        if (!$upd) {
            throw new Exception("Prepare failed: " . $con->error);
        }

        $upd->bind_param("s", $newClaimId);
        $upd->execute();
        $upd->close();

        $stmt->close();
        $con->commit();

        return $newClaimId;

    } catch (Exception $e) {
        $con->rollback();
        throw $e;
    }
}

// ===============================
// Example usage (uncomment to test)
// ===============================
// include("db_connection/database.php");
// echo generateClaimId($con);
?>
